<?php 
  class Customicons {
    private $stylesheet_path;
    private $cache_file;

    public function __construct() {
        $this->stylesheet_path = get_stylesheet_directory() . '/assets/css/icons.css';
        $this->cache_file = get_stylesheet_directory() . '/icons_cache.json';

        add_action('customize_register', array($this, 'customizer_controls'));
        add_action('customize_save_after', array($this, 'generate_font_stylesheet'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_custom_font_stylesheet'));
        add_action('customize_save_after', array($this, 'mytheme_check_custom_icons_option'));
        add_action('customize_save_after', array($this, 'clear_icons_cache'));
        add_action('after_switch_theme', array($this, 'clear_icons_cache'));
    }

    public function customizer_controls($wp_customize) {
        $wp_customize->add_section('custom_font_section', array(
            'title'    => __('Custom Icon Settings', 'text-domain'),
            'priority' => 30,
        ));

        $wp_customize->add_setting('enable_custom_icon_option', array(
            'default'           => false,
            'sanitize_callback' => 'sanitize_text_field',
        ));

        $wp_customize->add_control('enable_custom_icon_option', array(
            'label'       => __('Enable Custom Icon Option', 'text-domain'),
            'section'     => 'custom_font_section',
            'settings'    => 'enable_custom_icon_option',
            'type'        => 'checkbox',
        ));

        $wp_customize->add_setting('custom_icon_ttf_enable', array(
            'default'           => false,
            'sanitize_callback' => 'sanitize_text_field',
        ));

        $wp_customize->add_control('custom_icon_ttf_enable', array(
            'label'       => __('Enable this option to upload TTF/OTF Files', 'text-domain'),
            'description' => __('Enable this option before uploading TTF/OTF file. After uploading and publishing, disable this option.', 'text-domain'),
            'section'     => 'custom_font_section',
            'settings'    => 'custom_icon_ttf_enable',
            'type'        => 'checkbox',
            'active_callback' => array($this, 'enable_custom_icon_option'),
        ));

        $this->add_file_upload_control($wp_customize, 'icon_eot_file_upload', 'EOT Font File', array($this, 'enable_custom_icon_option'));
        $this->add_file_upload_control($wp_customize, 'icon_ttf_file_upload', 'TTF Font File', array($this, 'enable_custom_icon_option'));
        $this->add_file_upload_control($wp_customize, 'icon_woff_file_upload', 'WOFF Font File', array($this, 'enable_custom_icon_option'));
        $this->add_file_upload_control($wp_customize, 'icon_woff2_file_upload', 'WOFF2 Font File', array($this, 'enable_custom_icon_option'));
        $this->add_file_upload_control($wp_customize, 'icon_svg_file_upload', 'SVG Font File', array($this, 'enable_custom_icon_option'));

        $wp_customize->add_setting('custom_setting', array(
            'default'           => '',
            'sanitize_callback' => 'sanitize_textarea_field',
        ));

        $wp_customize->add_control('custom_setting', array(
            'label'    => __('Custom CSS', 'text-domain'),
            'section'  => 'custom_font_section',
            'settings' => 'custom_setting',
            'type'     => 'textarea',
            'active_callback' => array($this, 'enable_custom_icon_option'),
        ));
    }

    private function add_file_upload_control($wp_customize, $setting_id, $label, $active_callback) {
        $wp_customize->add_setting($setting_id, array(
            'default'           => '',
            'sanitize_callback' => 'esc_url_raw',
        ));
        $wp_customize->add_control(new WP_Customize_Upload_Control($wp_customize, $setting_id, array(
            'label'           => __($label, 'text-domain'),
            'section'         => 'custom_font_section',
            'settings'        => $setting_id,
            'active_callback' => $active_callback,
        )));
    }

    public function enable_custom_icon_option($control) {
        return $control->manager->get_setting('enable_custom_icon_option')->value() == true;
    }

    public function generate_font_stylesheet() {
        $font_eot  = get_theme_mod('icon_eot_file_upload', '');
        $font_ttf  = get_theme_mod('icon_ttf_file_upload', '');
        $font_woff = get_theme_mod('icon_woff_file_upload', '');
        $font_woff2 = get_theme_mod('icon_woff2_file_upload', '');
        $font_svg  = get_theme_mod('icon_svg_file_upload', '');
        $custom_css = get_theme_mod('custom_setting', '');

        $custom_icons_css = '';
        preg_match_all('/\.(.*?):before\s*{\s*content:\s*"\\\(.*?)";/m', $custom_css, $matches, PREG_SET_ORDER);
        foreach ($matches as $match) {
            $icon_class = 'custom-' . trim($match[1]);
            $icon_content = $match[2];
            $custom_icons_css .= ".{$icon_class}:before {
                content: \"\\{$icon_content}\";
            }\n";
        }

        $font_sources = [];

        if (!empty($font_eot)) {
            $font_sources[] = "url('{$font_eot}')";
            $font_sources[] = "url('{$font_eot}#iefix') format('embedded-opentype')";
        }
        if (!empty($font_ttf)) {
            $font_sources[] = "url('{$font_ttf}') format('truetype')";
        }
        if (!empty($font_woff)) {
            $font_sources[] = "url('{$font_woff}') format('woff')";
        }
        if (!empty($font_woff2)) {
            $font_sources[] = "url('{$font_woff2}') format('woff2')";
        }
        if (!empty($font_svg)) {
            $font_sources[] = "url('{$font_svg}#custom-icons') format('svg')";
        }

        $font_sources_str = implode(",\n", $font_sources);

        $font_css = "@font-face {
            font-family: 'custom-icons';
            src: {$font_sources_str};
            font-weight: normal;
            font-style: normal;
            font-display: block;
        }
        [class^='custom-'], [class*=' custom-'] {
            font-family: 'custom-icons' !important;
            speak: never;
            font-style: normal;
            font-weight: normal;
            font-variant: normal;
            text-transform: none;
            line-height: 1;
            -webkit-font-smoothing: antialiased;
            -moz-osx-font-smoothing: grayscale;
        }

        {$custom_icons_css}";

        $dir_path = dirname($this->stylesheet_path);
        if (!is_dir($dir_path)) {
            wp_mkdir_p($dir_path);
        }

        $file_writing_result = file_put_contents($this->stylesheet_path, $font_css);
        if ($file_writing_result === false) {
            error_log('Error writing to file: ' . $this->stylesheet_path);
        } else {
            $this->clear_icons_cache();
        }
    }

    public function enqueue_custom_font_stylesheet() {
        if (file_exists($this->stylesheet_path)) {
            wp_enqueue_style('custom-fonts', get_stylesheet_directory_uri() . '/assets/css/icons.css', array(), filemtime($this->stylesheet_path));
        }
    }

    public function mytheme_check_custom_icons_option() {
        $custom_fonts_enabled = get_theme_mod('custom_icon_ttf_enable', false);
        $config_file = ABSPATH . 'wp-config.php';
        $config_content = file_get_contents($config_file);
        $allow_unfiltered_uploads_defined = strpos($config_content, "define('ALLOW_UNFILTERED_UPLOADS'") !== false;
    
        // Add ALLOW_UNFILTERED_UPLOADS at the very end of the config file
        if ($custom_fonts_enabled && !$allow_unfiltered_uploads_defined) {
            $new_config_content = $config_content . "\n\n/** Allow unfiltered uploads for custom icons */\n" . "define('ALLOW_UNFILTERED_UPLOADS', true);\n";
            file_put_contents($config_file, $new_config_content);
        } elseif (!$custom_fonts_enabled && $allow_unfiltered_uploads_defined) {
            $new_config_content = str_replace("define('ALLOW_UNFILTERED_UPLOADS', true);\n", '', $config_content);
            file_put_contents($config_file, $new_config_content);
        }
    }
    

    public function clear_icons_cache() {
        if (file_exists($this->cache_file)) {
            unlink($this->cache_file);
        }
    }
}

new Customicons();